using System;
using System.Data;
using System.Collections;
using System.Reflection;
using System.Text.RegularExpressions;
using System.Text;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using DBCONSTANTS = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
	#region Medical device comment block
	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>6/5/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Defines the OrderedUnit object.</summary>
	#endregion
	//
	#region OrderedUnit
	public class OrderedUnit : BaseBusinessObject, IRecordStatusCode
	{
		#region Variables

		private DataTable _orderedUnitDataTable;
		private DataRow _orderedUnitDataRow;

		// Stores current crossmatch information:
		string _currentCrossmatchResultCode;
		string _currentCrossmatchTechId;
		DateTime _currentCrossmatchDate;

		#endregion


		#region Constructors

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1230"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>OrderedUnit object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="406"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Create an instance of OrderedUnit defaulting to a "new" object.
		/// </summary>
		public OrderedUnit()
		{
			GetOrderedUnitData(Guid.NewGuid(), string.Empty);
			System.Diagnostics.Debug.WriteLine("----------------------->Note that this is marked as a \"new\" OrderedUnit");
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4949"> 
		///		<ExpectedInput>Valid DataRow</ExpectedInput>
		///		<ExpectedOutput>OrderedUnit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4950"> 
		///		<ExpectedInput>Null DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///	<Developer>John Vrooland</Developer>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Create an instance of OrderedUnit from a data row passed by the user.
		/// Has to contain all referenced properties or an exception will occur.
		/// </summary>
		/// <param name="dr">DataRow with OrderedUnit data</param>
		public OrderedUnit(DataRow dr)
		{
			if (dr==null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("ordered unit").ResString);
			}

			_orderedUnitDataTable = dr.Table;
			_orderedUnitDataRow = dr;

			LoadCurrentCrossmatchData();
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4951"> 
		///		<ExpectedInput>Valid orderedUnitGuid and division code</ExpectedInput>
		///		<ExpectedOutput>An instance of OrderedUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4952"> 
		///		<ExpectedInput>Invalid arguuments</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Create an instance of OrderedUnit from a Guid passed by the user,
		/// with locking and read-only attribute specified.
		/// Specified item is retrieved from the database.
		/// </summary>
		/// <param name="orderedUnitGuid">OrderedUnit item Guid identifier</param>
		/// <param name="divisionCode">Division Code</param>
		/// <param name="readOnly">Indicates need for read only or read/write</param>
		public OrderedUnit(Guid  orderedUnitGuid,
			string divisionCode,
			bool readOnly) : this()
		{
			if (orderedUnitGuid == Guid.Empty || divisionCode == null || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("orderedUnitGuid and/or division code").ResString);
			}
			this.IsReadOnly = readOnly;
			GetOrderedUnitData(orderedUnitGuid, divisionCode);

			LoadCurrentCrossmatchData();
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4953"> 
		///		<ExpectedInput>Valid bloodUnitGuid and patientGuid</ExpectedInput>
		///		<ExpectedOutput>An instance of OrderedUnit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4954"> 
		///		<ExpectedInput>Invalid bloodUnitGuid and patientGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Create an instance of OrderedUnit from a Guid passed by the user with
		/// default read-only property. Specified item is retrieved from the database.
		/// </summary>
		/// <param name="bloodUnitGuid">OrderedUnit item Guid identifier</param>
		/// <param name="patientGuid">Division code</param>
		/// <summary>
		/// Constructor
		/// </summary>
		public OrderedUnit(Guid bloodUnitGuid, Guid patientGuid)
		{
			if (bloodUnitGuid == Guid.Empty || patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid and/or patietnguid").ResString);
			}
			// CR2899: new argument
			GetOrderedUnitData(bloodUnitGuid, patientGuid);

			LoadCurrentCrossmatchData();
		}		

		#endregion


		#region Properties

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7344"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7345"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns current crossmatch result code
		/// </summary>
		public string CurrentCrossmatchResultCode
		{
			get
			{
				return _currentCrossmatchResultCode;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7348"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7349"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns current crossmatch techid
		/// </summary>
		public string CurrentCrossmatchTechId
		{
			get
			{
				return _currentCrossmatchTechId;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7346"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7347"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return current crossmatch date
		/// </summary>
		public DateTime CurrentCrossmatchDate
		{
			get
			{
				return _currentCrossmatchDate;
			}
		}

		
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4955"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4956"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Guid value containing the  Ordered Unit Guid
		/// </summary>
		public Guid OrderedUnitGuid
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.OrderedUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (Guid)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.OrderedUnitGuid];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.OrderedUnitGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4957"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4958"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Guid value containing the  Ordered Component Guid
		/// </summary>
		public Guid OrderedComponentGuid
		{
			
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.OrderedComponentGuid))
				{
					return System.Guid.Empty;
				}
				return (Guid)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.OrderedComponentGuid];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.OrderedComponentGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4959"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4960"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Guid value containing the  Ordered Test Guid
		/// </summary>
		public Guid OrderedTestGuid
		{
			
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.OrderedTestGuid))
				{
					return System.Guid.Empty;
				}
				return (Guid)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.OrderedTestGuid];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.OrderedTestGuid] = value;
			}
		}
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4961"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4962"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// Guid value containing the  Blood Unit Guid
		/// </summary>
		public Guid BloodUnitGuid
		{
			
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.BloodUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (Guid)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.BloodUnitGuid];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.BloodUnitGuid] = value;
			}
		}
	
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4965"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4966"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// DateTime value containing the  Selected Date
		/// </summary>
		public DateTime SelectedDate
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.SelectedDate))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.SelectedDate];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.SelectedDate] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6418"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6419"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///		SelectedTechId
		/// </summary>
		public string SelectedTechId
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.SelectedTechId))
				{
					return string.Empty;
				}
				return (string)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.SelectedTechId];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.SelectedTechId] = value;
			}
		}
		
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4967"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4968"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// DateTime value containing the  Crossmatch Date
		/// WARNING!!! THIS DOES NOT RETURN THE MOST RECENT CrossmatchDate!  USE 
		/// CurrentCrossmatchDate FOR THIS!
		/// </summary>
		public DateTime CrossmatchDate
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchDate))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchDate];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchDate] = value;
			}
		}
		
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4969"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>CrossmatchResultCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4970"> 
		///		<ExpectedInput>CrossmatchResultCode</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// string value containing the Crossmatch Result Code
		/// WARNING!!! THIS DOES NOT RETURN THE MOST RECENT CrossmatchResultCode!  USE 
		/// CurrentCrossmatchResultCode FOR THIS!
		/// </summary>
		public string CrossmatchResultCode
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchResultCode))
				{
					return string.Empty;
				}
				return (string)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchResultCode];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchResultCode] = value;
			}
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4973"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>CrossmatchTechId</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4974"> 
		///		<ExpectedInput>CrossmatchTechId</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		///
		/// <summary>
		/// string value containing the  Crossmatch Tech ID
		/// WARNING!!! THIS DOES NOT RETURN THE MOST RECENT CrossmatchTechId!  USE 
		/// CurrentCrossmatchTechId FOR THIS!
		/// </summary>
		public string CrossmatchTechID
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchTechId))
				{
					return string.Empty;
				}
				return (string)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchTechId];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchTechId] = value;
			}
		}
		
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4971"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>CrossmatchRequiredIndicator</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4972"> 
		///		<ExpectedInput>CrossmatchRequiredIndicator</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// string value containing the  Crossmatch Result Code
		/// </summary>
		public bool CrossmatchRequiredIndicator
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator))
				{
					return false;
				}
				return (bool)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator] = value;
			}
		}
				
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4975"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Release Date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4976"> 
		///		<ExpectedInput>Release Date</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// DateTime value containing the  Release Date
		/// </summary>
		public DateTime ReleaseDate
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.ReleaseDate))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.ReleaseDate];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.ReleaseDate] = value;
			}
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7526"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7527"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReleaseTechId
		/// </summary>
		public string ReleaseTechId
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.ReleaseTechId))
				{
					return string.Empty;
				}
				return (string)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.ReleaseTechId];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.ReleaseTechId] = value;
			}
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4977"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Division Code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4978"> 
		///		<ExpectedInput>Division Code</ExpectedInput>
		///		<ExpectedOutput>none</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// string value containing the  Division Code
		/// </summary>
		public string DivisionCode
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.DivisionCode))
				{
					return string.Empty;
				}
				return (string)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.DivisionCode];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.DivisionCode] = value;
			}
		}
				
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4979"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>RecordStatusCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4980"> 
		///		<ExpectedInput>RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// string value containing the  Record Status Code
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.RecordStatusCode))
				{
					return Common.RecordStatusCode.Unknown;
				}
				return Common.Utility.GetRecordStatusCodeFromString(_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.RecordStatusCode].ToString());
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(value);
			}
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4981"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4982"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// DateTime value containing the  Last Update Date
		/// </summary>
		public new DateTime LastUpdateDate
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.LastUpdateDate))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.LastUpdateDate];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.LastUpdateDate] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4983"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Last update user</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4984"> 
		///		<ExpectedInput>Last update user</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// string value containing the  Last Update User
		/// </summary>
		public new string LastUpdateUser
		{
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.LastUpdateUser))
				{
					return "";
				}
				return (string)_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.LastUpdateUser];
			}
			set 
			{
				_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.LastUpdateUser] = value;
			}
		}
		
		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4985"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>RowVersion</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4986"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2003</CreationDate>
		/// <summary>
		/// timestamp value containing the  Row Version
		/// </summary>
		public new byte[] RowVersion
		{
			
			get 
			{
				if (_orderedUnitDataRow.IsNull(Common.VbecsTables.OrderedUnit.RowVersion))
				{
					return null;
				}
				return (byte[])_orderedUnitDataRow[Common.VbecsTables.OrderedUnit.RowVersion];
			}
		}
				
		#endregion


		#region Static Methods

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/18/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3738"> 
		///		<ExpectedInput>Valid Blood Unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Valid data table of ordered unit information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3739"> 
		///		<ExpectedInput>Invalid Blood Unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataTable of OrderUnit historical information based on a given BloodUnitGuid
		/// </summary>
		/// <param name="bloodUnitGuid">GUID</param>
		/// <param name="divisionCode">string</param>
		/// <returns>DataTable</returns>
		public static DataTable GetHistoricalOrderedUnits(System.Guid bloodUnitGuid, string divisionCode)
		{
			if (bloodUnitGuid == Guid.Empty || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid and/or divisionCode").ResString);
			}
			return DAL.OrderedUnit.GetHistoricalOrderedUnits(bloodUnitGuid, divisionCode);
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4987"> 
		///		<ExpectedInput>Valid Guid and Division Code</ExpectedInput>
		///		<ExpectedOutput>DataTable containing at least 1 row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4988"> 
		///		<ExpectedInput>Empty Guid and Division Code</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of ordered units for the component
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns></returns>
		public static DataTable GetOrderedUnitByComponent(System.Guid orderedComponentGuid, string divisionCode)
		{
			if (orderedComponentGuid == Guid.Empty || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("orderedComponentGuid and/or divisionCode").ResString);
			}

			return DAL.OrderedUnit.GetOrderedUnitByComponent(orderedComponentGuid, divisionCode);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4989"> 
		///		<ExpectedInput>Valid Guid and Division Code</ExpectedInput>
		///		<ExpectedOutput>Count > 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4990"> 
		///		<ExpectedInput>Invalid Guid and Division Code</ExpectedInput>
		///		<ExpectedOutput>Count == 0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the number of ordered units for the component
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns></returns>
		public static int GetOrderedUnitCount(System.Guid orderedComponentGuid, string divisionCode)
		{
			if (orderedComponentGuid == Guid.Empty || divisionCode == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("orderedComponentGuid and/or divisionCode").ResString);
			}

			return (int)DAL.OrderedUnit.GetOrderedUnitCount(orderedComponentGuid, divisionCode).Rows[0][0];
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4991"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>0 to many Rows returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4992"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of ordered units for the component
		/// </summary>
		/// <param name="patientGuid">patientGUID</param>
		/// <returns></returns>
		/// BR_73.02
		public static DataTable GetOrderedUnitsForPatient(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patientGuid").ResString);
			}

			return DAL.OrderedUnit.GetOrderedUnitsForPatient(patientGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4111"> 
		///		<ExpectedInput>Valid specimen Guid</ExpectedInput>
		///		<ExpectedOutput>ArrayList containing 0 to many rows</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4112"> 
		///		<ExpectedInput>Empty specimen Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of ordered units for a given specimen
		/// </summary>
		/// <param name="patientSpecimenGuid">Specimen unique identifier</param>
		/// <returns>Array list of specimen objects</returns>
		public static ArrayList GetOrderedUnitsForSpecimen(System.Guid patientSpecimenGuid)
		{
			if (patientSpecimenGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient specimen unique identifier").ResString);
			}
			//
			System.Data.DataTable dt = DAL.OrderedUnit.GetOrderedUnitsForSpecimen(patientSpecimenGuid);
			//
			ArrayList returnList = new ArrayList();
			//
			foreach(DataRow dr in dt.Rows)
			{
				returnList.Add((Guid)dr[Common.VbecsTables.OrderedUnit.OrderedUnitGuid]);
			}
			//
			return returnList;
		}

		///<Developers>
		///	<Developer>EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4993"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable with > 0 rows</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4994"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of ordered units for tag printing
		/// BR_43.01
		/// </summary>
		/// <returns></returns>
		public static DataTable GetOrderedUnitsForUnit()
		{
			return DAL.OrderedUnit.GetOrderedUnitsForUnit();
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2998"> 
		///		<ExpectedInput>Valid Ordered Test GUID or valid Ordered Component GUID</ExpectedInput>
		///		<ExpectedOutput>Valid datatable of all ordered units for repeat XM</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2999"> 
		///		<ExpectedInput>Invalid Ordered Test GUID and invalid Ordered Component GUID</ExpectedInput>
		///		<ExpectedOutput>Empty datatable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all Ordered Units that are crossmatched based on 
		/// OrderedTest GUID or OrderComponent GUID.
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <returns>datatable</returns>
		public static DataTable GetOrderedUnitsForRepeatXM(Guid orderedTestGuid, Guid orderedComponentGuid)
		{
			return DAL.OrderedUnit.GetOrderedUnitsForRepeatXM(orderedTestGuid, orderedComponentGuid);
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3000"> 
		///		<ExpectedInput>none</ExpectedInput>
		///		<ExpectedOutput>An empty ordered units test datatable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="407"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates an empty OrderedUnit table
		/// </summary>
		/// <returns>datatable</returns>
		public static DataTable InitializeOrderedUnitTable()
		{
			DataTable dtOrderedUnit = new DataTable(Common.VbecsTables.OrderedUnit.TableName);

			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.OrderedUnitGuid, typeof(Guid));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.OrderedComponentGuid, typeof(Guid)); //
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.OrderedTestGuid, typeof(Guid));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.BloodUnitGuid, typeof(Guid));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.SelectedDate, typeof(DateTime));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.SelectedTechId, typeof(string));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator, typeof(bool)); //
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.CrossmatchDate, typeof(DateTime));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.CrossmatchResultCode, typeof(string));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.CrossmatchTechId, typeof(string));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.ReleaseDate, typeof(DateTime)); //
			//dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.ReleaseReasonID, typeof(int)); //
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.DivisionCode, typeof(string));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.RecordStatusCode, typeof(string));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.LastUpdateUser, typeof(string));
			dtOrderedUnit.Columns.Add(Common.VbecsTables.OrderedUnit.LastUpdateDate, typeof(DateTime));
			
			return dtOrderedUnit;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3024"> 
		///		<ExpectedInput>PatientGuid, BloodUnit Guid</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3025"> 
		///		<ExpectedInput>Invalid Bloodunit Guid</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a boolean stating if incompatible crossmatch results exist between patient and blood unit
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="bloodUnitGUID"></param>
		/// <returns></returns>
		public static bool IncompatibleCrossmatchResultsExist(System.Guid patientGuid, System.Guid bloodUnitGUID)
		{
			return DAL.OrderedUnit.IncompatibleCrossmatchResultsExist(patientGuid, bloodUnitGUID);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1231"> 
		///		<ExpectedInput>Valid Blood Unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Valid data table of ordered unit information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1837"> 
		///		<ExpectedInput>Invalid Blood Unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataTable of OrderUnit information based on a given BloodUnitGuid
		/// </summary>
		/// <param name="bloodUnitGuid">GUID</param>
		/// <param name="divisionCode">string</param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrderedUnit(System.Guid bloodUnitGuid, string divisionCode)
		{
			return DAL.OrderedUnit.GetOrderedUnit(bloodUnitGuid, divisionCode);
		}

		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2004</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="4018"> 
		///		<ExpectedInput>ArrayList, DateTime, TechId</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4345"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Releases asigned unit(s)
		/// Implements BR_47.07
		/// </summary>
		/// <param name="assignedUnitInfo">ArrayList containing DataRows selected for processing</param>
		/// <param name="releaseDate">date assignment is released</param>
		/// <param name="releaseTechId">user releasing assignment</param>
		/// <param name="releaseComment">user comments stating reason for release</param>
		/// <param name="expiredUnits">list of expired units, used for client info purposes</param>
		/// <param name="workloadProcessId">workload process to tie workload to</param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns>success/failure</returns>
		public static bool ReleaseOrderedUnits(ArrayList assignedUnitInfo, System.DateTime releaseDate, string releaseTechId, string releaseComment, ref ArrayList expiredUnits, Common.WorkloadProcessID workloadProcessId, DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			bool result = false;
			ArrayList bloodUnitGuidList = new ArrayList(assignedUnitInfo.Count);
			DataTable dtOrderedUnit = GetEmptyOrderedUnitDataTable();
			//
			for (int idx = 0; idx < assignedUnitInfo.Count; idx++)
			{
				Hashtable record = (Hashtable)assignedUnitInfo[idx];
				//
				Guid bloodUnitGuid = new Guid( (string)record[VbecsTables.BloodUnit.BloodUnitGuid] );
				Guid orderedUnitGuid = new Guid( (string)record[VbecsTables.OrderedUnit.OrderedUnitGuid] );
				//
				BloodUnit bloodUnit = new BloodUnit( bloodUnitGuid );
				OrderedUnit orderedUnit = new OrderedUnit( orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode, false );
				//
				DataRow drOrderedUnit = dtOrderedUnit.NewRow();
				drOrderedUnit[TABLES.OrderedUnit.OrderedUnitGuid] = orderedUnitGuid;
				drOrderedUnit[TABLES.OrderedUnit.ReleaseTechId] = releaseTechId;
				drOrderedUnit[TABLES.OrderedUnit.ReleaseDate] = releaseDate;
				drOrderedUnit[TABLES.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drOrderedUnit[TABLES.OrderedUnit.RowVersion] = orderedUnit.RowVersion;
				drOrderedUnit[TABLES.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				//
				// Check whether unit is expired, if so add to array list 
				// for informational purposes; the unit is still released
				if ( bloodUnit.BloodUnitMedia.UnitExpirationDate <= VBECSDateTime.GetDivisionCurrentDateTime() )
				{
					Object id = ( bloodUnit.EyeReadableUnitId );
					if ( !expiredUnits.Contains(id) )
					{
						expiredUnits.Add(id);
					}
				}
				//
				dtOrderedUnit.Rows.Add( drOrderedUnit );
				//
				// Keep track of blood units released
				if( !bloodUnitGuidList.Contains( bloodUnitGuid ) )
				{
					bloodUnitGuidList.Add( bloodUnitGuid );
				}
			}
			//
			// Generate the workload event data
			Guid [] bloodUnitGuids = new Guid[bloodUnitGuidList.Count];
			bloodUnitGuidList.CopyTo( bloodUnitGuids );
			DataTable dtWorkload = WorkloadEvent.GenerateWorkloadData( workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false) ;
			//
			// Invoke call to remove assignment(s)
			if ( dtOrderedUnit.Rows.Count > 0 )
			{
				result = DAL.OrderedUnit.ReleaseOrderedUnits( dtOrderedUnit, dtWorkload, releaseComment, Common.UpdateFunction.UC047FrmReleaseUnits, dtBCEMessages );
			}
			//
			return result;
		}

		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/13/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7245"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7246"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Releases any ordered units for a given blood unit
		/// Implements BR_11.05
		/// </summary>
		/// <param name="bloodUnit">parent blood unit</param>
		/// <param name="releaseComment">unit status comment describing release details</param>
		/// <param name="releaseDate">date assignment is released</param>
		/// <param name="releaseTechId">user releasing assignment</param>
		/// <param name="updateFunctionId">update function</param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns>success/failure</returns>
		public static bool ReleaseOrderedUnits( BloodUnit bloodUnit, string releaseComment, System.DateTime releaseDate, string releaseTechId, UpdateFunction updateFunctionId, DataTable dtBCEMessages )
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			DataTable orderedUnits = OrderedUnit.GetOrderedUnit( bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );
			DataTable dtOrderedUnit = GetEmptyOrderedUnitDataTable();
			//
			for ( int idx = 0; idx < orderedUnits.Rows.Count; idx++ )
			{
				DataRow drOrderedUnit = dtOrderedUnit.NewRow();
				drOrderedUnit[TABLES.OrderedUnit.BloodUnitGuid] = orderedUnits.Rows[idx][TABLES.OrderedUnit.BloodUnitGuid];
				drOrderedUnit[TABLES.OrderedUnit.ReleaseTechId] = releaseTechId;
				drOrderedUnit[TABLES.OrderedUnit.ReleaseDate] = releaseDate;
				drOrderedUnit[TABLES.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drOrderedUnit[TABLES.OrderedUnit.RowVersion] = orderedUnits.Rows[idx][TABLES.OrderedUnit.RowVersion];
				drOrderedUnit[TABLES.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				dtOrderedUnit.Rows.Add( drOrderedUnit );
			}
			//
			return DAL.OrderedUnit.ReleaseOrderedUnits ( dtOrderedUnit, updateFunctionId, dtBCEMessages );
		}

		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/20/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7309"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7310"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets DataTable for release of patient assignment consisting of 
		/// ordered units for a given blood unit
		/// </summary>
		/// <param name="bloodUnitArray">arraylist of blood units</param>
		/// <param name="releaseDate">date assignment is released</param>
		/// <param name="releaseTechId">user releasing assignment</param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrderedUnitDataTableForReleaseOfPatientAssignment( ArrayList bloodUnitArray, DateTime releaseDate, string releaseTechId )
		{
			DataTable dtOrderedUnit = GetEmptyOrderedUnitDataTable();
			//
			for ( int i = 0; i < bloodUnitArray.Count; i++ )
			{
				BOL.BloodUnit bloodUnit = (BOL.BloodUnit)bloodUnitArray[i];
				DataTable orderedUnits = OrderedUnit.GetOrderedUnit( bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode );

				for ( int j = 0; j < orderedUnits.Rows.Count; j++ )
				{
					DataRow drOrderedUnit = dtOrderedUnit.NewRow();
					drOrderedUnit[TABLES.OrderedUnit.BloodUnitGuid] = orderedUnits.Rows[j][TABLES.OrderedUnit.BloodUnitGuid];
					drOrderedUnit[TABLES.OrderedUnit.ReleaseTechId] = releaseTechId;
					drOrderedUnit[TABLES.OrderedUnit.ReleaseDate] = releaseDate;
					drOrderedUnit[TABLES.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
					drOrderedUnit[TABLES.OrderedUnit.RowVersion] = orderedUnits.Rows[j][TABLES.OrderedUnit.RowVersion];
					drOrderedUnit[TABLES.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtOrderedUnit.Rows.Add( drOrderedUnit );
				}
			}
			//
			return dtOrderedUnit;
		}

		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4019"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4344"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets all the assigned/crossmatched units in the user's division
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetAssignedUnits()
		{
			DataTable assignedUnits = DAL.OrderedUnit.GetOrderedUnits();

			ArrayList records = new ArrayList(assignedUnits.Rows.Count);
		
			foreach (System.Data.DataRow dr in assignedUnits.Rows)
			{
				Hashtable recordMap = new Hashtable(14);
				recordMap.Add(TABLES.Patient.PatientFirstName, dr[TABLES.Patient.PatientFirstName].ToString());
				recordMap.Add(TABLES.Patient.PatientMiddleName, dr[TABLES.Patient.PatientMiddleName].ToString());
				recordMap.Add(TABLES.Patient.PatientLastName, dr[TABLES.Patient.PatientLastName].ToString());
				recordMap.Add(TABLES.Patient.DisplayVistaPatientId, dr[TABLES.Patient.DisplayVistaPatientId].ToString());
				recordMap.Add(TABLES.BloodUnit.EyeReadableUnitId, dr[TABLES.BloodUnit.EyeReadableUnitId].ToString());
				recordMap.Add(DBCONSTANTS.FullProductCode, dr[DBCONSTANTS.FullProductCode].ToString());
				recordMap.Add(TABLES.BloodProduct.ProductShortName, dr[TABLES.BloodProduct.ProductShortName].ToString());
				recordMap.Add(DBCONSTANTS.ABORh, dr[DBCONSTANTS.ABORh].ToString());
				recordMap.Add(TABLES.BloodUnitMedia.UnitExpirationDate, dr[TABLES.BloodUnitMedia.UnitExpirationDate].ToString());
				recordMap.Add(TABLES.DonationType.DonationTypeId, dr[TABLES.DonationType.DonationTypeId].ToString());
				recordMap.Add(TABLES.DonationType.DonationTypeText, dr[TABLES.DonationType.DonationTypeText].ToString());
				recordMap.Add(TABLES.PatientSpecimen.SpecimenExpirationDate, dr[TABLES.PatientSpecimen.SpecimenExpirationDate].ToString());
				recordMap.Add(TABLES.OrderedUnit.OrderedUnitGuid, dr[TABLES.OrderedUnit.OrderedUnitGuid].ToString());
				recordMap.Add(TABLES.BloodUnit.BloodUnitGuid, dr[TABLES.BloodUnit.BloodUnitGuid].ToString());
				records.Add(recordMap);
			}

			return records;
		}


		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2004</CreationDate>
		///<TestCases>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets empty OrderedUnit DataTable
		/// </summary>
		/// <returns>DataTable</returns>
		internal static DataTable GetEmptyOrderedUnitDataTable()
		{
			DataTable dtOrderedUnit = new DataTable(TABLES.OrderedUnit.TableName);
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.BloodUnitGuid, typeof(System.Guid));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.OrderedUnitGuid, typeof(System.Guid));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.ReleaseTechId, typeof(string));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.ReleaseDate, typeof(System.DateTime));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.LastUpdateUser, typeof(string));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.RowVersion, typeof(byte[]));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.DivisionCode, typeof(string));
			return dtOrderedUnit;
		}

		///<Developers>
		///<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2004</CreationDate>
		///<TestCases>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets empty UnitStatus DataTable
		/// </summary>
		/// <returns>DataTable</returns>
		private static DataTable GetEmptyUnitStatusDataTable()
		{
			DataTable dtUnitStatus = new DataTable();
			
			dtUnitStatus.Columns.Add(TABLES.BloodUnitStatus.BloodUnitGuid, typeof(System.Guid));
			dtUnitStatus.Columns.Add(TABLES.BloodUnitStatusCodeCurrent.UnitStatusCode, typeof(string));
			dtUnitStatus.Columns.Add(TABLES.BloodUnitStatus.UnitStatusComments, typeof(string));
			dtUnitStatus.Columns.Add(TABLES.BloodUnitStatus.LastUpdateUser, typeof(string));
			dtUnitStatus.Columns.Add(TABLES.BloodUnitStatus.RowVersion, typeof(byte[]));

			DataColumn[] keys = {dtUnitStatus.Columns[TABLES.BloodUnitStatus.BloodUnitGuid]};
			dtUnitStatus.PrimaryKey = keys;

			return dtUnitStatus;
		}

		#endregion

		#region Methods

		private void LoadCurrentCrossmatchData()
		{
			if (this.OrderedUnitGuid != Guid.Empty)
			{
				DataRow dtRow = DAL.OrderedUnit.GetCurrentCrossmatchData(this.OrderedUnitGuid);
				if (dtRow != null)
				{
					if (!dtRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchResultCode))
					{
						_currentCrossmatchResultCode = (string) dtRow[Common.VbecsTables.OrderedUnit.CrossmatchResultCode];
					}
					if (!dtRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchTechId))
					{
						_currentCrossmatchTechId = (string) dtRow[Common.VbecsTables.OrderedUnit.CrossmatchTechId];
					}
					if (!dtRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchDate))
					{
						_currentCrossmatchDate = (DateTime) dtRow[Common.VbecsTables.OrderedUnit.CrossmatchDate];
					}
				}
				else
				{
					_currentCrossmatchResultCode = string.Empty;
					_currentCrossmatchTechId = string.Empty;
					_currentCrossmatchDate = DateTime.MinValue;
				}
			}
		}

		/// <summary>
		/// Retrieves the OrderedUnit from the DAL
		/// </summary>
		/// <param name="orderedUnitGuid">OrderedUnit item to get Guid identifier</param>
		/// <param name="divisionCode">Division Code</param>
		private void GetOrderedUnitData(System.Guid orderedUnitGuid, string divisionCode)
		{
			_orderedUnitDataTable = DAL.OrderedUnit.GetOrderedUnitByGuid(orderedUnitGuid, 
				divisionCode);
			if (_orderedUnitDataTable.Rows.Count > 0) 
			{
				_orderedUnitDataRow = _orderedUnitDataTable.Rows[0];
			}
			else 
			{
				MakeNew();
			}
			_orderedUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

		/// <summary>
		/// Retrieves the OrderedUnit from the DAL
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		private void GetOrderedUnitData(Guid bloodUnitGuid, Guid patientGuid)
		{
			// CR2899: new argument
			_orderedUnitDataTable = DAL.OrderedUnit.GetOrderedUnitByPatientAndUnit(bloodUnitGuid.ToString(), 
				patientGuid);
			if (_orderedUnitDataTable.Rows.Count > 0) 
			{
				_orderedUnitDataRow = _orderedUnitDataTable.Rows[0];
			}
			else 
			{
				MakeNew();
			}
			_orderedUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

	
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/15/2010</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="9153"> 
		///		<ExpectedInput>Valid Blood Unit unique identifier string</ExpectedInput>
		///		<ExpectedOutput>Valid data table of ordered unit information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9154"> 
		///		<ExpectedInput>Empty Blood Unit unique identifiers string</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implementation of gov.va.med.vbecs.BOL.BaseBusinessObject.LoadDataRowFromThis
		/// <param name="bloodUnitGuidList"></param>
		/// <param name="patientGuid"></param>
		/// </summary>
		/// CR2899: This is a new method to accomodate retrieving OrderedUnits in batch.
		public static DataTable GetOrderedUnitData(string bloodUnitGuidList, Guid patientGuid)
		{
			return DAL.OrderedUnit.GetOrderedUnitByPatientAndUnit(bloodUnitGuidList, 
				patientGuid);
		}
		

		/// <summary>
		/// Create a new instance of OrderedUnit data
		/// </summary>
		private void MakeNew()
		{
			this.IsNew = true;
			this.IsReadOnly = false;
			//
			_orderedUnitDataRow = _orderedUnitDataTable.NewRow();
			//_orderedUnitDataTable.Rows.Add(_orderedUnitDataRow); (Best done in Save)
			//
			this.OrderedUnitGuid = Guid.NewGuid();
			this.LastUpdateUser = Common.LogonUser.LogonUserName;
		}

		/// <summary>
		/// Handler for dirty flag and thwarting attempts to modify a non-writable object
		/// </summary>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			if (this.IsReadOnly && !this.IsNew) 
			{
				throw new BusinessObjectException("OrderedUnit object is marked read-only");
			}
			this.IsDirty = true;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7528"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7529"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implementation of gov.va.med.vbecs.BOL.BaseBusinessObject.LoadDataRowFromThis
		/// <param name="dtRow"></param>
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.BloodUnitGuid))
			{
				dtRow[Common.VbecsTables.OrderedUnit.BloodUnitGuid] = this.BloodUnitGuid; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchDate))
			{
				dtRow[Common.VbecsTables.OrderedUnit.CrossmatchDate] = this.CrossmatchDate; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator))
			{
				dtRow[Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator] = this.CrossmatchRequiredIndicator; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchResultCode))
			{
				dtRow[Common.VbecsTables.OrderedUnit.CrossmatchResultCode] = this.CrossmatchResultCode; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchTechId))
			{
				dtRow[Common.VbecsTables.OrderedUnit.CrossmatchTechId] = this.CrossmatchTechID; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.DivisionCode))
			{
				dtRow[Common.VbecsTables.OrderedUnit.DivisionCode] = this.DivisionCode; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.LastUpdateDate))
			{
				dtRow[Common.VbecsTables.OrderedUnit.LastUpdateDate] = this.LastUpdateDate; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.LastUpdateUser))
			{
				dtRow[Common.VbecsTables.OrderedUnit.LastUpdateUser] = this.LastUpdateUser; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.OrderedUnitGuid))
			{
				dtRow[Common.VbecsTables.OrderedUnit.OrderedUnitGuid] = this.OrderedUnitGuid; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.OrderedComponentGuid))
			{
				dtRow[Common.VbecsTables.OrderedUnit.OrderedComponentGuid] = this.OrderedComponentGuid; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.SelectedDate))
			{
				dtRow[Common.VbecsTables.OrderedUnit.SelectedDate] = this.SelectedDate; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.SelectedTechId))
			{
				dtRow[Common.VbecsTables.OrderedUnit.SelectedTechId] = this.SelectedTechId; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.ReleaseDate))
			{
				dtRow[Common.VbecsTables.OrderedUnit.ReleaseDate] = this.ReleaseDate; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.ReleaseTechId))
			{
				dtRow[Common.VbecsTables.OrderedUnit.ReleaseTechId] = this.ReleaseTechId; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.RecordStatusCode))
			{
				dtRow[Common.VbecsTables.OrderedUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.RowVersion))
			{
				dtRow[Common.VbecsTables.OrderedUnit.RowVersion] = this.RowVersion; 
			}

			return dtRow;
		}

		/// <summary>
		/// Implementation of gov.va.med.vbecs.BOL.BaseBusinessObject.LoadFromDataRow
		/// <param name="dtRow"></param>
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.BloodUnitGuid))
			{
				this.BloodUnitGuid = (Guid) dtRow[Common.VbecsTables.OrderedUnit.BloodUnitGuid]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchDate))
			{
				this.CrossmatchDate = (DateTime) dtRow[Common.VbecsTables.OrderedUnit.CrossmatchDate]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator))
			{
				this.CrossmatchRequiredIndicator = (bool) dtRow[Common.VbecsTables.OrderedUnit.CrossmatchRequiredIndicator]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchResultCode))
			{
				this.CrossmatchResultCode = dtRow[Common.VbecsTables.OrderedUnit.CrossmatchResultCode].ToString(); 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.CrossmatchTechId))
			{
				this.CrossmatchTechID = dtRow[Common.VbecsTables.OrderedUnit.CrossmatchTechId].ToString(); 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.DivisionCode))
			{
				this.DivisionCode = dtRow[Common.VbecsTables.OrderedUnit.DivisionCode].ToString(); 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.LastUpdateDate))
			{
				this.LastUpdateDate = (DateTime) dtRow[Common.VbecsTables.OrderedUnit.LastUpdateDate]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.LastUpdateUser))
			{
				this.LastUpdateUser = dtRow[Common.VbecsTables.OrderedUnit.LastUpdateUser].ToString(); 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.OrderedUnitGuid))
			{
				this.OrderedUnitGuid = (Guid) dtRow[Common.VbecsTables.OrderedUnit.OrderedUnitGuid]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.OrderedComponentGuid))
			{
				this.OrderedComponentGuid = (Guid) dtRow[Common.VbecsTables.OrderedUnit.OrderedComponentGuid]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.SelectedDate))
			{
				this.SelectedDate = (DateTime) dtRow[Common.VbecsTables.OrderedUnit.SelectedDate]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.SelectedTechId))
			{
				this.SelectedTechId = (string) dtRow[Common.VbecsTables.OrderedUnit.SelectedTechId]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.ReleaseDate))
			{
				this.ReleaseDate = (DateTime) dtRow[Common.VbecsTables.OrderedUnit.ReleaseDate]; 
			}

			if(dtRow.Table.Columns.Contains(Common.VbecsTables.OrderedUnit.ReleaseTechId))
			{
				this.ReleaseTechId = (string) dtRow[Common.VbecsTables.OrderedUnit.ReleaseTechId]; 
			}
	
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/2/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9202"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9203"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2941
		/// CR 3015: added orderedComponentGuid and issueToLocationName
		/// </summary>
		/// <param name="bloodUnitGuid"><see cref="Guid"/></param>
		/// <param name="patientGuid"><see cref="Guid"/></param>
		/// <param name="orderedComponentGuid"><see cref="Guid"/></param>
		/// <param name="issueToLocationName"><see cref="string"/></param>
		/// <param name="dispenseStatus"><see cref="HL7BPDispenseStatus"/></param>
		/// <param name="electronicCrossmatch"><see cref="bool"/></param>
		public static DataTable UpdateBce(Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocationName, HL7BPDispenseStatus dispenseStatus, bool electronicCrossmatch)
		{
			// CR 2945: adding support for transactions
			DataTable dtMessageLog = VbecsBpsMessage.ConstructMessage(bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, dispenseStatus, electronicCrossmatch);
			dtMessageLog.TableName = TABLES.MessageLog.TableName;
			return dtMessageLog;
		}

		///<Developers>
		///	<Developer>Department of Veterans Affairs</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9208"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9209"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 3015 Added this method for all cases when we send RE message to BCE. Since the OrderedUnit record exists for this patient and unit
		/// I can get OrderedComponentGuid inside this method instead of repeating this code in many different use cases.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="electronicCrossmatch"></param>
		/// <returns></returns>
		public static DataTable UpdateBceForRelease(Guid bloodUnitGuid, Guid patientGuid, bool electronicCrossmatch)
		{
			OrderedUnit orderedUnit = new OrderedUnit(bloodUnitGuid, patientGuid);
			DataTable dtMessageLog = VbecsBpsMessage.ConstructMessage(bloodUnitGuid, patientGuid, orderedUnit.OrderedComponentGuid, "", HL7BPDispenseStatus.RE, electronicCrossmatch);
			dtMessageLog.TableName = TABLES.MessageLog.TableName;
			return dtMessageLog;
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/2/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9204"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9205"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases> 
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2941
		/// CR 3015: added orderedComponentGuid and issueToLocationName
		/// </summary>
		/// <param name="bloodUnitGuid"><see cref="Guid"/></param>
		/// <param name="patientGuid"><see cref="Guid"/></param>
		/// <param name="orderedComponentGuid"><see cref="Guid"/></param>
		/// <param name="issueToLocationName"><see cref="string"/></param>
		/// <param name="dispenseStatus"><see cref="HL7BPDispenseStatus"/></param>
		public static void UpdateAndSaveBce(Guid bloodUnitGuid, Guid patientGuid, Guid orderedComponentGuid, string issueToLocationName, HL7BPDispenseStatus dispenseStatus)
		{
			VbecsBpsMessage.ConstructAndSendMessage(bloodUnitGuid, patientGuid, orderedComponentGuid, issueToLocationName, dispenseStatus);
		}

		///<Developers>
		///	<Developer>Department of Veterans Affairs</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9210"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9211"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets PatientGuid based on OrderedUnitGuid
		/// Added for BCE
		/// CR 2941
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <returns></returns>
		public static Guid GetPatientForOrderedUnit(Guid orderedUnitGuid)
		{
			DataTable dtPatient = DAL.OrderedUnit.GetPatientForOrderedUnit(orderedUnitGuid);
			if (dtPatient.Rows.Count > 0)
			{
				return (Guid)dtPatient.Rows[0][VbecsTables.Patient.PatientGuid];
			}
			else
				return Guid.Empty;
		}

		#endregion
        
	}

	#endregion
}
